//////////////////////////////////////////////////////////////////////////////
//
//  Microsoft Research Singularity / Netstack
//
//  Copyright (c) Microsoft Corporation.  All rights reserved.
//
//  File: TestDhcpClient.cs
//
//  Note:
//       This is a minimal test of Dns packet assembly and
//       parsing.  It only tests good input to date.
//

using System;
using System.Diagnostics;

using System.Net.IP;
using Drivers.Net;
using NetStack.Protocols;
using Dns = NetStack.Protocols.Dns;

namespace NetStack.Protocols.TestDnsClient
{
    public class Test
    {
        private static readonly byte [][] rawPackets = new byte [][] {
            // A small selection of good DNS frame dumps
            new byte [] {
                0x00,0x01,0x01,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
                0x01,0x33,0x01,0x30,0x03,0x32,0x34,0x34,0x03,0x32,0x30,0x39,
                0x07,0x69,0x6e,0x2d,0x61,0x64,0x64,0x72,0x04,0x61,0x72,0x70,
                0x61,0x00,0x00,0x0c,0x00,0x01,
            },
            new byte [] {
                0x00,0x01,0x81,0x80,0x00,0x01,0x00,0x01,0x00,0x00,0x00,0x00,
                0x01,0x33,0x01,0x30,0x03,0x32,0x34,0x34,0x03,0x32,0x30,0x39,
                0x07,0x69,0x6e,0x2d,0x61,0x64,0x64,0x72,0x04,0x61,0x72,0x70,
                0x61,0x00,0x00,0x0c,0x00,0x01,0xc0,0x0c,0x00,0x0c,0x00,0x01,
                0x00,0x00,0x09,0x2a,0x00,0x16,0x09,0x72,0x65,0x73,0x6f,0x6c,
                0x76,0x65,0x72,0x31,0x06,0x6c,0x65,0x76,0x65,0x6c,0x33,0x03,
                0x6e,0x65,0x74,0x00,
            },
            new byte [] {
                0x00,0x02,0x01,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
                0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x03,0x63,
                0x6f,0x6d,0x00,0x00,0x0f,0x00,0x01,
            },
            new byte [] {
                0x00,0x02,0x81,0x80,0x00,0x01,0x00,0x03,0x00,0x00,0x00,0x00,
                0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x03,0x63,
                0x6f,0x6d,0x00,0x00,0x0f,0x00,0x01,0xc0,0x0c,0x00,0x0f,0x00,
                0x01,0x00,0x00,0x00,0xf0,0x00,0x0a,0x00,0x0a,0x05,0x6d,0x61,
                0x69,0x6c,0x61,0xc0,0x0c,0xc0,0x0c,0x00,0x0f,0x00,0x01,0x00,
                0x00,0x00,0xf0,0x00,0x0a,0x00,0x0a,0x05,0x6d,0x61,0x69,0x6c,
                0x62,0xc0,0x0c,0xc0,0x0c,0x00,0x0f,0x00,0x01,0x00,0x00,0x00,
                0xf0,0x00,0x0a,0x00,0x0a,0x05,0x6d,0x61,0x69,0x6c,0x63,0xc0,
                0x0c,
            },
            new byte [] {
                0x00,0x03,0x01,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
                0x03,0x77,0x77,0x77,0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,
                0x66,0x74,0x03,0x63,0x6f,0x6d,0x00,0x00,0x01,0x00,0x01,
            },
            new byte [] {
                0x00,0x03,0x81,0x80,0x00,0x01,0x00,0x02,0x00,0x00,0x00,0x00,
                0x03,0x77,0x77,0x77,0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,
                0x66,0x74,0x03,0x63,0x6f,0x6d,0x00,0x00,0x01,0x00,0x01,0xc0,
                0x0c,0x00,0x05,0x00,0x01,0x00,0x00,0x0b,0xa3,0x00,0x1d,0x03,
                0x77,0x77,0x77,0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,0x66,
                0x74,0x03,0x63,0x6f,0x6d,0x05,0x6e,0x73,0x61,0x74,0x63,0x03,
                0x6e,0x65,0x74,0x00,0xc0,0x2f,0x00,0x01,0x00,0x01,0x00,0x00,
                0x01,0x19,0x00,0x04,0xcf,0x2e,0x9c,0x9c,
            },
            new byte [] {
                0x00,0x04,0x01,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
                0x03,0x77,0x77,0x77,0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,
                0x66,0x74,0x03,0x63,0x6f,0x6d,0x05,0x6e,0x73,0x61,0x74,0x63,
                0x03,0x6e,0x65,0x74,0x00,0x00,0x01,0x00,0x01,
            },
            new byte [] {
                0x00,0x04,0x81,0x80,0x00,0x01,0x00,0x01,0x00,0x00,0x00,0x00,
                0x03,0x77,0x77,0x77,0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,0x6f,
                0x66,0x74,0x03,0x63,0x6f,0x6d,0x05,0x6e,0x73,0x61,0x74,0x63,
                0x03,0x6e,0x65,0x74,0x00,0x00,0x01,0x00,0x01,0xc0,0x0c,0x00,
                0x01,0x00,0x01,0x00,0x00,0x01,0x09,0x00,0x04,0xcf,0x2e,0x9c,
                0x9c,
            },
            new byte [] {
                0x00,0x05,0x01,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
                0x03,0x77,0x77,0x77,0x07,0x68,0x6f,0x74,0x6d,0x61,0x69,0x6c,
                0x03,0x63,0x6f,0x6d,0x00,0x00,0x01,0x00,0x01,
            },
            new byte [] {
                0x00,0x05,0x81,0x80,0x00,0x01,0x00,0x04,0x00,0x00,0x00,0x00,
                0x03,0x77,0x77,0x77,0x07,0x68,0x6f,0x74,0x6d,0x61,0x69,0x6c,
                0x03,0x63,0x6f,0x6d,0x00,0x00,0x01,0x00,0x01,0xc0,0x0c,0x00,
                0x05,0x00,0x01,0x00,0x00,0x08,0x2a,0x00,0x1b,0x03,0x77,0x77,
                0x77,0x07,0x68,0x6f,0x74,0x6d,0x61,0x69,0x6c,0x03,0x63,0x6f,
                0x6d,0x05,0x6e,0x73,0x61,0x74,0x63,0x03,0x6e,0x65,0x74,0x00,
                0xc0,0x2d,0x00,0x01,0x00,0x01,0x00,0x00,0x08,0x2a,0x00,0x04,
                0xcf,0x44,0xab,0xe9,0xc0,0x2d,0x00,0x01,0x00,0x01,0x00,0x00,
                0x08,0x2a,0x00,0x04,0xcf,0x44,0xac,0xef,0xc0,0x2d,0x00,0x01,
                0x00,0x01,0x00,0x00,0x08,0x2a,0x00,0x04,0xcf,0x44,0xad,0xf5,
            },
            new byte [] {
                0x00,0x02,0x85,0x83,0x00,0x01,0x00,0x00,0x00,0x01,0x00,0x00,
                0x08,0x6f,0x68,0x6f,0x64,0x73,0x6f,0x6e,0x32,0x00,0x00,0x01,
                0x00,0x01,0x00,0x00,0x06,0x00,0x01,0x00,0x00,0x0e,0x10,0x00,
                0x3d,0x0a,0x63,0x6f,0x72,0x70,0x2d,0x64,0x63,0x2d,0x30,0x36,
                0x04,0x63,0x6f,0x72,0x70,0x09,0x6d,0x69,0x63,0x72,0x6f,0x73,
                0x6f,0x66,0x74,0x03,0x63,0x6f,0x6d,0x00,0x07,0x6d,0x61,0x72,
                0x69,0x6f,0x72,0x6f,0xc0,0x35,0x00,0x00,0x07,0xaa,0x00,0x00,
                0x0e,0x10,0x00,0x00,0x02,0x58,0x00,0x01,0x51,0x80,0x00,0x00,
                0x0e,0x10,
            },
            new byte [] {
                0x00,0x03,0x81,0x80,0x00,0x01,0x00,0x01,0x00,0x00,0x00,0x00,
                0x03,0x32,0x31,0x31,0x03,0x32,0x35,0x34,0x02,0x35,0x35,0x03,
                0x31,0x35,0x37,0x07,0x49,0x4e,0x2d,0x41,0x44,0x44,0x52,0x04,
                0x41,0x52,0x50,0x41,0x00,0x00,0x0c,0x00,0x01,0xc0,0x0c,0x00,
                0x0c,0x00,0x01,0x00,0x00,0x02,0x50,0x00,0x27,0x0a,0x72,0x65,
                0x64,0x2d,0x64,0x6e,0x73,0x2d,0x30,0x31,0x07,0x72,0x65,0x64,
                0x6d,0x6f,0x6e,0x64,0x04,0x63,0x6f,0x72,0x70,0x09,0x6d,0x69,
                0x63,0x72,0x6f,0x73,0x6f,0x66,0x74,0x03,0x63,0x6f,0x6d,0x00,
            },
        };

        public static void Dump(byte [] data)
        {
            for (int i = 0; i < data.Length; i += 16) {
                Console.Write("{0:x4}  ", i);
                int rowEnd = Math.Min(16, data.Length - i) + i;

                // Hex dump
                for (int j = i; j != rowEnd; j++)
                    Console.Write("{0:x2} ", data[j]);
                for (int j = rowEnd; (j % 16) != 0; j++)
                    Console.Write("   ");
                Console.Write(" ");

                // ASCII dump
                for (int j = i; j < rowEnd; j++) {
                    if (data[j] > 31 && data[j] < 128)
                        Console.Write("{0}", (char)data[j]);
                    else
                        Console.Write(".");
                }
                Console.Write("\n");
            }
        }

        public static void TestByteParsing()
        {
            int packetCount = rawPackets.Length / rawPackets.Rank;
            for (int i = 0; i < packetCount; i++) {
                Console.WriteLine("-- Packet {0} ---", i);
                Dump(rawPackets[i]);
                int offset = 0;
                Dns.Format format = Dns.Format.Parse(rawPackets[i],
                                                     ref offset);
                Debug.Assert(offset == rawPackets[i].Length ||
                             format.GetRCode() != Dns.RCode.NoError);
                Debug.Assert(format == format);
                // format.Size may not equal offset here, because
                // packet may have RFC1035 compression enabled.
                // Debug.Assert(offset == format.Size)
                offset = 0;
                byte [] rendered = new byte [format.Size];
                format.Write(rendered, ref offset);

                offset = 0;
                Dns.Format restored = Dns.Format.Parse(rendered, ref offset);
                Debug.Assert(format.GetHashCode() == restored.GetHashCode());
            }
        }

        public static void TestLabelEncoding()
        {
            string [] passes = new string [] {
                "a", "foo", "moo-ba.lah-lah-lah.com",
                "icicle.infernal.fiery.hell.net",
                "1.2.222.221.in-addr.arpa",
                // 63 characters between separators
                // 3 separators
                // 4 * 63 + 3 = 255 characters
                "abcdefghij0123456789abcdefghij0123456789" +
                "abcdefghij0123456789abc" + "." +
                "abcdefghij0123456789abcdefghij0123456789" +
                "abcdefghij0123456789abc" + "." +
                "abcdefghij0123456789abcdefghij0123456789" +
                "abcdefghij0123456789abc" + "." +
                "abcdefghij0123456789abcdefghij0123456789" +
                "abcdefghij0123456789abc"
            };

            foreach (string original in passes) {
                byte [] encoded = Dns.LabelEncoding.GetBytes(original);
                string decoded;
                int used = Dns.LabelEncoding.GetString(encoded, 0,
                                                       out decoded);
                int expected = Dns.LabelEncoding.GetByteCount(original);
                Debug.Assert(used == expected);
                Console.WriteLine("\"{0}\" \"{1}\" {2}",
                                  original, decoded,
                                  (original != decoded) ? "*" : " ");
            }
        }

        public static void Main()
        {
            TestLabelEncoding();
            TestByteParsing();
        }
    }
} // NetStack.Runtime.TestDnsClient
